'use strict';

angular.module('genisis').service('request', ['api', '$rootScope', '$location',
  'user',
  '$window',
  function ($api, $rootScope, $location, $user, $window) {
    var _requests = [];
    var currentCopiedRequest = {
      associatedRequests: [],
      notify: false,
      source: null,
      type: null,
      studyApproval: null,
      description: null,
      createdBy: $user.username(),
      requestTypeId: null,
      sourceId: null,
      studyApprovalId: null,
      title: null,
      statusDescription: 'Draft'
    };

    this.canChangeCommentType = function () {
      return $user.destinationMgr() || $user.Administrator();
    };

    this.canCopyRequest = function (status) {
      if ((status === 'Draft') || ($user.sourceMgr())) {
        return false;
      } else {
        return true;
      }
    };

    this.canReturnToDDM = function (status) {
      if ((status === 'Sent' || status === 'RequestAccepted') &&
        ($user.sourceMgr() || $user.Administrator())) {
        return true;
      } else {
        return false;
      }
    };

    this.copyRequest = function (request) {
      currentCopiedRequest.description = request.description;
      currentCopiedRequest.requestTypeId = request.requestTypeId;
      currentCopiedRequest.sourceId = request.sourceId;
      currentCopiedRequest.studyApprovalId = request.studyApprovalId;
      currentCopiedRequest.title = 'Copy of - ' + request.title;
      $location.path('/copyRequest');
    };

    this.getCopiedRequest = function () {
      if (currentCopiedRequest.title) {
        return currentCopiedRequest;
      } else {
        return {
          associatedRequests: [],
          notify: false,
          source: null,
          type: null,
          studyApproval: null,
          description: null,
          createdBy: $user.username()
        };
      }
    };

    this.mustComment = function (request) {
      $window.alert(
        'You must add a comment before proceeding with this action'
      );
      $location.path('/requests/' + request.id);
    };

    this.canAccess = function (request, user) {
      var accessibleStatus = ['Sent', 'RequestAccepted',
        'RequestNotAccepted'
      ];
      if (user.sourceMgr()) {
        if (accessibleStatus.indexOf(request.statusDescription) > -1) {
          return true;
        } else {
          return false;
        }
      } else if (user.requester()) {
        if (request.createdBy === user.username()) {
          return true;
        } else {
          return false;
        }
      } else if (user.Administrator() || user.destinationMgr()) {
        return true;
      } else {
        return false;
      }
    };

    this.getCommentType = function (user) {
      // If Requester default it to general
      // If DDM default to operation
      // If DSM default it to operation
      // Else defaut it general

      if (user.requester()) {
        return 'general';
      } else if (user.destinationMgr() || user.sourceMgr()) {
        return 'operation';
      } else {
        return 'general';
      }
    };

    this.cancellable = function (req) {
      if (!$user.sourceMgr() && (
          req.statusDescription === 'Submitted' ||
          req.statusDescription === 'Returned' ||
          req.statusDescription === 'Sent' ||
          req.statusDescription === 'RequestAccepted' ||
          req.statusDescription === 'ResultsDelivered'
        )) {
        return true;
      } else {
        return false;
      }
    };

    /**
     * Return the name of the request given the id
     * @param int id the id number of the source
     * @return string source name for that id
     **/
    this.toName = function (id) {
      if (!id) {
        return 'None';
      }

      var request = this.find(id);
      return (request) ? request.title : 'None';
    };

    /**
     * Load all requests or specific requests for a user
     * @param int uid user id
     * @return array{} list of requests that were loaded
     **/
    this.load = function (payload) {
      return $api.retrieve('requests', payload).then(
        function (request) {
          if (request.success) {

            if (request.response instanceof Array) {
              _requests = request.response;
            } else {
              _requests = [request.response];
            }

            return _requests;
          }

          return [];
        }
      );
    };

    /**
     * A helper method to parse logs into a specified format
     * based on requirements
     **/

    this.parseLogs = function (logs) {
      var logArray = [];
      if (logs && logs.length > 0) {
        // Loop through each log object.
        for (var i = 0; i < logs.length; i++) {
          // Since the server is returning one log object plus
          // commentHistory which is another array...we have to
          // check which comment is part of this action too.
          var log = logs[i];
          var primaryLog = {};
          // If it is Draft, we should just use createdBy and
          // CreatedOn else use modifiedOn
          if (log.statusDescription === 'Draft') {
            primaryLog.statusDescription = log.statusDescription;
            primaryLog.createdBy = log.createdBy;
            primaryLog.createdOn = log.createdOn;
          } else {
            primaryLog.statusDescription = log.statusDescription;
            primaryLog.modifiedBy = log.modifiedBy;
            primaryLog.modifiedOn = log.modifiedOn;
          }
          // Since we set primaryLog main information...let's check
          // for comment, we will have to parse through the array
          // CommentHistory and check type: 'action', if there is one
          // add this commentHistory's comment to this primaryLog
          // and remove it from the list

          // Define commentHistory
          var commentHistory = log.commentHistory;
          if (commentHistory && commentHistory.length > 0) {
            // Are there any commentHistory objects with type 'action'?
            var commentAction = commentHistory.filter(function (e) {
              return (e.type === 'action');
            });
            // Should only be one....
            if (commentAction && commentAction.length === 1) {
              primaryLog.comment = commentAction[0].comments;
              primaryLog.commentType = commentAction[0].commentType;
            }

            logArray.push(primaryLog);

            var commentLogs = commentHistory.filter(function (e) {
              return (e.type !== 'action');
            });

            if (commentLogs && commentLogs.length > 0) {
              for (var j = 0; j < commentLogs.length; j++) {
                var newLog = {};
                var commentLog = commentLogs[j];
                newLog.statusDescription = 'Comment';
                newLog.comment = commentLog.comments;
                newLog.commentType = commentLog.commentType;
                newLog.createdBy = commentLog.createdBy;
                newLog.createdOn = commentLog.createdOn;
                logArray.push(newLog);
              }
            }
          } else {
            logArray.push(primaryLog);
          }
        }
        return logArray;
      } else {
        return null;
      }
    };

    /**
     * Load the logs for a specific request
     * @param int request id
     * @return array{} list of requests that were loaded
     **/
    this.logs = function (requestID, userRole) {

      var logs = [];

      return $api.retrieve('requests/' + requestID + '/' + userRole +
          '/log', {})
        .then(
          function (request) {
            if (request.success) {

              if (request.response instanceof Array) {
                logs = request.response;
              } else {
                logs = [request.response];
              }
              return logs;
            }

            return [];
          }
        );
    };

    /**
     * Find a specific request in the list
     * Note: requests must have already been loaded
     * @param int id the request id
     * @return Object|null
     **/
    this.find = function (id) {

      if (!id) {
        return null;
      }

      //make sure it's always an integer
      id = parseInt(id, 10);

      var req = null;

      angular.forEach(_requests, function (request) {
        if (request.id === id) {
          req = request;
        }
      });

      return req;
    };

    this.findByID = function (id) {
      //make sure it's always an integer
      id = parseInt(id, 10);

      if (!id || id < 1) {
        return {};
      }

      return $api.retrieve('requests/' + id).then(
        function (request) {
          if (request.success) {
            return request.response;
          } else {
            return {};
          }
        }
      );
    };

    /**
     * Return the array of all loaded requests
     * @return array{}
     **/
    this.get = function () {
      return _requests;
    };

    /**
     * Validate the request
     * @param Object the request that needs validated
     * @return boolean
     **/
    this.valid = function (request) {

      if (!request.title) {
        // $rootScope.errors.push('You must enter a request title.');
        $window.alert('You must enter a request title.');
        return false;
      }

      if (!request.sourceId) {
        // $rootScope.errors.push('You must select a source.');
        $window.alert('You must select a source');
        return false;
      }

      if (!request.studyApprovalId) {
        // $rootScope.errors.push('You must select an approved study.');
        $window.alert('You must select an approved study.');
        return false;
      }

      if (!request.requestTypeId) {
        // $rootScope.errors.push(
        //   'You must select a request type.');
        $window.alert('You must select a request type.');

        return false;
      }

      // if (typeof request.notify !== 'boolean') {
      //   // $rootScope.errors.push(
      //   //   'You must select whether or not to notify collaborators.');
      //   $window.alert(
      //     'You must select whether or not to notify collaborators.');

      //   return false;
      // }

      if (!request.description) {
        // $rootScope.errors.push(
        //   'You must enter a request description.');
        $window.alert('You must enter a request description.');

        return false;
      }

      return true;
    };


    /**
     * Validate the draft request (before submitted)
     * @param Object the request that needs validated
     * @return boolean
     **/
    this.validDraft = function (request) {

      if (!request.title) {
        // $rootScope.errors.push('You must enter a request title.');
        $window.alert('You must enter a request title.');
        return false;
      }

      return true;
    };

    this.exceededMaxCharacters = function (msg, option) {
      switch (option) {
      case 'title':
        if (msg && msg.length > 256) {
          return true;
        } else {
          return false;
        }
        break;
      default:
        if (msg && msg.length > 3000) {
          return true;
        } else {
          return false;
        }
        break;
      }
    };
  }
]);